{ BFG - Buchhaltung fr Geeks - bookkeeping for geeks

  Copyright (C) 2004 Peter Gerwinski <peter@gerwinski.de>

  BFG is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published
  by the Free Software Foundation, version 2.

  BFG is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
  License for more details.

  You should have received a copy of the GNU General Public License
  along with BFG; see the file COPYING. If not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA. }

program BFG;

uses
  GPC;

const
  BFGVersion = '0.9.1';

type
  VATModeType = (NoVAT, Net, Gross, AntiNet, AntiGross);
  SignModeType = (WithSign, WithoutSign, WithMacro);

type
  PAliasRec = ^TAliasRec;
  PEventRec = ^TEventRec;
  POpRec = ^TOpRec;
  PAccountRec = ^TAccountRec;

  TAliasRec = record
    Name, Meaning: PString;
    Next: PAliasRec
  end;

  TEventRec = record
    Account, AntiAccount: PAccountRec;
    Amount: Real;
    Date, Comment: PString;
    DayNumber: Integer;
    Next: PEventRec
  end;

  TOpRec = record
    OpCode: Char;
    OtherAccount: PAccountRec;
    Next: POpRec
  end;

  TAccountRec = record
    LongName, ShortName: PString;
    SectionFlag, TeXFlag, OpeningFlag, StoneAgeFlag: Boolean;
    CollectVAT: Real;
    Events: PEventRec;
    Operations: POpRec;
    Next: PAccountRec
  end;

var
  OutFileName: TString = '-';
  GnuPlotFileName: TString = '';
  Aliases: PAliasRec = nil;
  Accounts: PAccountRec = nil;
  DateFormat: TString = '%Y-%m-%d';
  DecimalPointFormat: TString = '.';
  ThousandSepFormat: TString = '';
  CurrentDate, CurrentRep, CurrentRepCount, CurrentRepUntil: TString = '';
  CurrentDayNumber: Integer = 0;
  DefaultVAT: Real = 0.16;
  Force, SummaryOnly, CurrentOnly,
    TeX, GnuPlot, OutFileTemp, GnuPlotFileTemp: Boolean = False;
  TimePeriod, DataSpec: TString = '';
  MinDayNumber: Integer = -MaxInt;
  MaxDayNumber: Integer = MaxInt;
  ErrorDetected: Boolean = False;
  MainFileName: TString;

procedure Error (const Message, InFileName: String; LineNo: Integer);
begin
  if InFileName = '' then
    Write (StdErr, ParamStr (0))
  else if InFileName = '-' then
    Write (StdErr, 'line ', LineNo)
  else
    Write (StdErr, InFileName, ':', LineNo);
  WriteLn (StdErr, ': ', Message);
  ErrorDetected := True
end;

function Round2 (x: Real): Real;
begin
  Round2 := 0.01 * Round (100 * x)
end;

procedure Init;
const
  LongOptions: array [1..10] of OptionType =
    (('output', RequiredArgument, nil, 'o'),
     ('summary', NoArgument, nil, 's'),
     ('current', NoArgument, nil, 'c'),
     ('tex', NoArgument, nil, 'x'),
     ('time-period', RequiredArgument, nil, 't'),
     ('data', RequiredArgument, nil, 'd'),
     ('gnuplot', OptionalArgument, nil, 'g'),
     ('force', NoArgument, nil, 'f'),
     ('help', NoArgument, nil, 'h'),
     ('version', NoArgument, nil, 'V'));
var
  LongIndex, i, d, y, D0, D1, M0, M1: Integer;
  TPYear, TPMonth, TPQuarter: Integer = -1;
  Ch: Char;
  T: TString;
  CurrentDate: TimeStamp;

  procedure OutputVersion;
  begin
    WriteLn ('BFG - Buchhaltung fr Geeks - bookkeeping for geeks, version ',
             BFGVersion);
    WriteLn ('Copyright (C) 2004 Peter Gerwinski - http://www.peter.gerwinski.de');
    WriteLn ('BFG is free software under the GNU GPL - see the file COPYING.')
  end;

  procedure OutputHelp;
  begin
    OutputVersion;
    WriteLn ('
Usage: bfg [options] [file]

Options:
  -o, --output=<file>
      Name of output file (default: stdout)
  -s, --summary
      Don''t output the calculations; just show the results.
  -c, --current
      Suppress output of calculations with the same comment that
      sum up to zero.
  -x, --tex
      Format the output as (La)TeX source.
  -t, --time-period=<specification>
      Take only events into account which fall into the specified
      period. In <specification>, a four-digit number specifies a
      year, a smaller number a month, a roman number a quarter of a
      year. (Examples: -t 2004, -t 2004-3, -t i/2004)
  -d, --data=<list>
      <List> is a comma-separated list of account names.
      Produce a file that contains the daily values of the given
      accounts as functions of the time (days from the Epoch) in
      a format suitable as input for Gnuplot.
  -g, --gnuplot[=<file>]
      Together with `--data'': Call Gnuplot to display the data.
      If <file> is given, it is used to store the commands given
      to Gnuplot.
  -f, --force
      Do not suppress output in case of an error.
  -h, --help
      Output this help and exit.
  -V, --version
      Output version information and exit.

For a reference of the input language, see the file REFERENCE.');
  end;

begin
  LongIndex := -1;
  repeat
    Ch := GetOptLong ('', LongOptions, LongIndex, False);
    case Ch of
      'o': OutFileName := OptionArgument;
      's': SummaryOnly := True;
      'c': CurrentOnly := True;
      'x': TeX := True;
      'd': DataSpec := OptionArgument + ',';
      't': TimePeriod := OptionArgument;
      'g': begin
             GnuPlot := True;
             if OptionArgument = '' then
               begin
                 GnuPlotFileTemp := True;
                 GnuPlotFileName := GetTempFileName
               end
           end;
      'f': Force := True;
      'h': begin
             OutputHelp;
             Halt (0)
           end;
      'V': begin
             OutputVersion;
             Halt (0)
           end;
    end
  until Ch = EndOfOptions;
  if FirstNonOption <= ParamCount then
    MainFileName := ParamStr (FirstNonOption)
  else
    MainFileName := '-';
  if GnuPlot and (OutFileName = '-') then
    begin
      OutFileTemp := True;
      OutFileName := GetTempFileName
    end;
  if TimePeriod <> '' then
    begin
      i := 1;
      while i <= Length (TimePeriod) do
        if TimePeriod[i] in ['0'..'9'] then
          begin
            T := '';
            while (i <= Length (TimePeriod))
                  and (TimePeriod[i] in ['0'..'9']) do
              begin
                T := T + TimePeriod[i];
                Inc (i)
              end;
            if Length (T) > 2 then
              ReadStr (T, TPYear)
            else
              ReadStr (T, TPMonth)
          end
        else if TimePeriod[i] = 'i' then
          begin
            T := '';
            while (i <= Length (TimePeriod))
                  and (TimePeriod[i] in ['i', 'v']) do
              begin
                T := T + TimePeriod[i];
                Inc (i)
              end;
            if T = 'i' then
              TPQuarter := 1
            else if T = 'ii' then
              TPQuarter := 2
            else if T = 'iii' then
              TPQuarter := 3
            else if T = 'iv' then
              TPQuarter := 4
          end
        else if TimePeriod[i] = 'I' then
          begin
            T := '';
            while (i <= Length (TimePeriod))
                  and (TimePeriod[i] in ['I', 'V']) do
              begin
                T := T + TimePeriod[i];
                Inc (i)
              end;
            if T = 'I' then
              TPQuarter := 1
            else if T = 'II' then
              TPQuarter := 2
            else if T = 'III' then
              TPQuarter := 3
            else if T = 'IV' then
              TPQuarter := 4
          end
        else
          Inc (i);
      if TPYear = -1 then
        begin
          GetTimeStamp (CurrentDate);
          TPYear := CurrentDate.Year
        end;
      if TPQuarter <> -1 then
        begin
          M0 := 3 * TPQuarter - 2;
          M1 := 3 * TPQuarter
        end
      else if TPMonth <> -1 then
        begin
          M0 := TPMonth;
          M1 := TPMonth
        end
      else
        begin
          M0 := 1;
          M1 := 12
        end;
      D0 := 1;
      D1 := MonthLength (M1, TPYear);
      d := 0;
      for y := 1970 to TPYear - 1 do
        begin
          Inc (d, 365);
          if IsLeapYear (y) then
            Inc (d)
        end;
      MinDayNumber := d + GetDayOfYear (D0, M0, TPYear);
      MaxDayNumber := d + GetDayOfYear (D1, M1, TPYear)
    end
end;

procedure Done;
var
  F: Text;
begin
  if OutFileTemp then
    begin
      Assign (F, OutFileName);
      Erase (F)
    end;
  if GnuPlotFileTemp then
    begin
      Assign (F, GnuPlotFileName);
      Erase (F)
    end;
  if ErrorDetected then
    Halt (-1)
end;

function FindAccount (const Name: String): PAccountRec;
var
  A: PAccountRec;
begin
  A := Accounts;
  while A <> nil do
    begin
      if ((A^.ShortName <> nil) and (Name = A^.ShortName^))
         or ((A^.LongName <> nil) and (Name = A^.LongName^)) then
        begin
          FindAccount := A;
          Exit
        end;
      A := A^.Next
    end;
  FindAccount := nil
end;

function TeXStr (const A: String) = S: TString;
var
  p: Integer;
begin
  S := A;
  p := Pos (' - ', S);
  while p <> 0 do
    begin
      Insert ('-', S, p + 1);
      p := PosFrom (' - ', S, p + 1)
    end;
  p := 1;
  while p <= Length (S) do
    begin
      case S[p] of
        '_': S[p] := ' ';
        '&', '%', '$':
          begin
            Insert ('\', S, p);
            Inc (p)
          end;
      end;
      Inc (p)
    end
end;

function TeXAmount (const A: String; ShowZero: Boolean;
                    SignMode: SignModeType) = S: TString;
var
  Test: Real;
  p, Code: Integer;
  Sign: Char;
begin
  if not ShowZero then
    begin
      Val (A, Test, Code);
      if (Code = 0) and (Test = 0) then
        begin
          S := '';
          Exit
        end
    end;
  S := A;
  p := pos ('.', S);
  if p = 0 then
    begin
      p := Length (S) + 1;
      S := S + '.00'
    end;
  Delete (S, p, 1);
  Insert (DecimalPointFormat, S, p);
  Dec (p, 3);
  while (p > 1) and (S[p-1] in ['0'..'9']) do
    begin
      Insert (ThousandSepFormat, S, p);
      Dec (p, 3)
    end;
  if S[1] in ['+', '-'] then
    case SignMode of
      WithSign:
        begin
          Insert ('$', S, 2);
          Insert ('$', S, 1)
        end;
      WithoutSign:
        Delete (S, 1, 1);
      WithMacro:
        begin
          Sign := S[1];
          Delete (S, 1, 1);
          if Sign = '+' then
            S := '\plus{' + S + '}'
          else
            S := '\minus{' + S + '}'
        end;
    end
end;

function TeXAmountNum (a: Real; ShowZero: Boolean;
                       SignMode: SignModeType): TString;
var
  S: TString;
begin
  WriteStr (S, a : 0 : 2);
  TeXAmountNum := TeXAmount (S, ShowZero, SignMode)
end;

function AccountTitleStr (Account: PAccountRec) = Title: TString;
begin
  if Account = nil then
    Title := ''
  else
    with Account^ do
      begin
        if (ShortName <> nil) and (LongName <> nil) then
          Title := ShortName^ + ' - ' + LongName^
        else if ShortName <> nil then
          Title := ShortName^
        else if LongName <> nil then
          Title := LongName^
        else
          Title := ''
      end
end;

function OpeningEvent (Event: PEventRec): Boolean;
begin
  OpeningEvent := (Event <> nil)
                  and (Event^.Account^.OpeningFlag
                       or Event^.AntiAccount^.OpeningFlag)
end;

function AccountSum (Account: PAccountRec) = Sum: Real;
var
  Op: POpRec;
  Event: PEventRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - AccountSum (Op^.OtherAccount)
          else
            Sum := Sum + AccountSum (Op^.OtherAccount);
          Op := Op^.Next
        end
    end
  else
    begin
      Event := Account^.Events;
      while Event <> nil do
        begin
          Sum := Sum + Event^.Amount;
          Event := Event^.Next
        end
    end
end;

function AccountSumMinus (Account: PAccountRec): Real;
forward;

function AccountSumPlus (Account: PAccountRec) = Sum: Real;
var
  Op: POpRec;
  Event: PEventRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - AccountSumMinus (Op^.OtherAccount)
          else
            Sum := Sum + AccountSumPlus (Op^.OtherAccount);
          Op := Op^.Next
        end
    end
  else
    begin
      Event := Account^.Events;
      while Event <> nil do
        begin
          if (Event^.Amount > 0) and not OpeningEvent (Event) then
            Sum := Sum + Event^.Amount;
          Event := Event^.Next
        end
    end
end;

function AccountSumMinus (Account: PAccountRec) = Sum: Real;
var
  Op: POpRec;
  Event: PEventRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - AccountSumPlus (Op^.OtherAccount)
          else
            Sum := Sum + AccountSumMinus (Op^.OtherAccount);
          Op := Op^.Next
        end
    end
  else
    begin
      Event := Account^.Events;
      while Event <> nil do
        begin
          if (Event^.Amount < 0) and not OpeningEvent (Event) then
            Sum := Sum + Event^.Amount;
          Event := Event^.Next
        end
    end
end;

function AccountStoneAgeSumPlus (Account: PAccountRec) = Sum: Real;
var
  Op: POpRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - Min (0, AccountSum (Op^.OtherAccount))
          else
            Sum := Sum + Max (0, AccountSum (Op^.OtherAccount));
          Op := Op^.Next
        end
    end
  else
    Sum := AccountSumPlus (Account)
end;

function AccountStoneAgeSumMinus (Account: PAccountRec) = Sum: Real;
var
  Op: POpRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - Max (0, AccountSum (Op^.OtherAccount))
          else
            Sum := Sum + Min (0, AccountSum (Op^.OtherAccount));
          Op := Op^.Next
        end
    end
  else
    Sum := AccountSumMinus (Account)
end;

function AccountSumOpening (Account: PAccountRec) = Sum: Real;
var
  Op: POpRec;
  Event: PEventRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - AccountSumOpening (Op^.OtherAccount)
          else
            Sum := Sum + AccountSumOpening (Op^.OtherAccount);
          Op := Op^.Next
        end
    end
  else
    begin
      Event := Account^.Events;
      while Event <> nil do
        begin
          if OpeningEvent (Event) then
            Sum := Sum + Event^.Amount;
          Event := Event^.Next
        end
    end
end;

function AccountSumMinusOpening (Account: PAccountRec) = Sum: Real;
forward;

function AccountSumPlusOpening (Account: PAccountRec) = Sum: Real;
var
  Op: POpRec;
  Event: PEventRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - Min (0, AccountSumOpening (Op^.OtherAccount))
          else
            Sum := Sum + Max (0, AccountSumOpening (Op^.OtherAccount));
          Op := Op^.Next
        end
    end
  else
    begin
      Event := Account^.Events;
      while Event <> nil do
        begin
          if (Event^.Amount > 0) and OpeningEvent (Event) then
            Sum := Sum + Event^.Amount;
          Event := Event^.Next
        end
    end
end;

function AccountSumMinusOpening (Account: PAccountRec) = Sum: Real;
var
  Op: POpRec;
  Event: PEventRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - Max (0, AccountSumOpening (Op^.OtherAccount))
          else
            Sum := Sum + Min (0, AccountSumOpening (Op^.OtherAccount));
          Op := Op^.Next
        end
    end
  else
    begin
      Event := Account^.Events;
      while Event <> nil do
        begin
          if (Event^.Amount < 0) and OpeningEvent (Event) then
            Sum := Sum + Event^.Amount;
          Event := Event^.Next
        end
    end
end;

function AccountSumAt (Account: PAccountRec; DayNumber: Integer) = Sum: Real;
var
  Op: POpRec;
  Event: PEventRec;
begin
  Sum := 0;
  if Account = nil then
    Exit;
  if Account^.Operations <> nil then
    begin
      Op := Account^.Operations;
      while Op <> nil do
        begin
          if Op^.OpCode = '-' then
            Sum := Sum - AccountSumAt (Op^.OtherAccount, DayNumber)
          else
            Sum := Sum + AccountSumAt (Op^.OtherAccount, DayNumber);
          Op := Op^.Next
        end
    end
  else
    begin
      Event := Account^.Events;
      while Event <> nil do
        begin
          if Event^.DayNumber <= DayNumber then
            Sum := Sum + Event^.Amount;
          Event := Event^.Next
        end
    end
end;

function AccountSumStr (Account: PAccountRec) = S: TString;
var
  Sum: Real;
begin
  Sum := AccountSum (Account);
  if Sum >= 0 then
    WriteStr (S, '+', Sum : 0 : 2)
  else
    WriteStr (S, Sum : 0 : 2)
end;

function AccountSumPlusStr (Account: PAccountRec) = S: TString;
var
  Sum: Real;
begin
  Sum := AccountSumPlus (Account);
  if Sum < 0 then
    begin
      Error ('internal error: positive sum is negative', '', 0);
      WriteStr (S, Sum : 0 : 2)
    end
  else
    WriteStr (S, '+', Sum : 0 : 2)
end;

function AccountSumMinusStr (Account: PAccountRec) = S: TString;
var
  Sum: Real;
begin
  Sum := AccountSumMinus (Account);
  if Sum > 0 then
    begin
      Error ('internal error: negative sum is positive', '', 0);
      WriteStr (S, '+', Sum : 0 : 2)
    end
  else if Sum = 0 then
    WriteStr (S, '-', Sum : 0 : 2)
  else
    WriteStr (S, Sum : 0 : 2)
end;

function AccountStoneAgeSumPlusStr (Account: PAccountRec) = S: TString;
var
  Sum: Real;
begin
  Sum := AccountStoneAgeSumPlus (Account);
  if Sum < 0 then
    begin
      Error ('internal error: positive sum is negative', '', 0);
      WriteStr (S, Sum : 0 : 2)
    end
  else
    WriteStr (S, '+', Sum : 0 : 2)
end;

function AccountStoneAgeSumMinusStr (Account: PAccountRec) = S: TString;
var
  Sum: Real;
begin
  Sum := AccountStoneAgeSumMinus (Account);
  if Sum > 0 then
    begin
      Error ('internal error: negative sum is positive', '', 0);
      WriteStr (S, '+', Sum : 0 : 2)
    end
  else if Sum = 0 then
    WriteStr (S, '-', Sum : 0 : 2)
  else
    WriteStr (S, Sum : 0 : 2)
end;

function AccountSumOpeningStr (Account: PAccountRec) = S: TString;
var
  Sum: Real;
begin
  Sum := AccountSumOpening (Account);
  if Sum >= 0 then
    WriteStr (S, '+', Sum : 0 : 2)
  else
    WriteStr (S, Sum : 0 : 2)
end;

function AccountSumPlusOpeningStr (Account: PAccountRec) = S: TString;
var
  Sum: Real;
begin
  Sum := AccountSumPlusOpening (Account);
  if Sum >= 0 then
    WriteStr (S, '+', Sum : 0 : 2)
  else
    begin
      Error ('internal error: positive sum is negative', '', 0);
      WriteStr (S, Sum : 0 : 2)
    end
end;

function AccountSumMinusOpeningStr (Account: PAccountRec) = S: TString;
var
  Sum: Real;
begin
  Sum := AccountSumMinusOpening (Account);
  if Sum > 0 then
    begin
      Error ('internal error: negative sum is positive', '', 0);
      WriteStr (S, '+', Sum : 0 : 2)
    end
  else if Sum = 0 then
    WriteStr (S, '-', Sum : 0 : 2)
  else
    WriteStr (S, Sum : 0 : 2)
end;

procedure ReverseLists;
var
  Account, PrevAccount, NextAccount: PAccountRec;
  Event, PrevEvent, NextEvent: PEventRec;
begin
  Account := Accounts;
  PrevAccount := nil;
  while Account <> nil do
    begin
      NextAccount := Account^.Next;
      Account^.Next:= PrevAccount;
      PrevAccount := Account;
      Event := Account^.Events;
      PrevEvent := nil;
      while Event <> nil do
        begin
          NextEvent := Event^.Next;
          Event^.Next:= PrevEvent;
          PrevEvent := Event;
          Event := NextEvent
        end;
      Account^.Events := PrevEvent;
      Account := NextAccount
    end;
  Accounts := PrevAccount
end;

procedure IncludeFile (const InFileName: String);
var
  InFile: Text;
  Line: TString;
  LineNo: Integer = 0;
  
  procedure ParseDate (const DateStr, Format: String; (* MARKED *)
                      var Year, Month, Day: Integer);
  var
    pd, pf: Integer;
  
    procedure ExtractInteger (const S: String; var p, x: Integer);
    var
      i: Integer;
    begin
      i := p;
      while (i <= Length (S)) and (S[i] in ['0'..'9']) do
        Inc (i);
      ReadStr (SubStr (S, p, i - p), x);
      p := i
    end;
  
  begin
    Year := 0;
    Month := 0;
    Day := 0;
    pd := 1;
    pf := 1;
    while (pd <= Length (DateStr)) and (pf <= Length (Format)) do
      begin
        if Format[pf] = '%' then
          begin
            if (pf + 1 > Length (Format))
              or not (Format[pf + 1] in ['%', 'Y', 'm', 'd']) then
              begin
                Error ('`%'' must be followed by `%'', `Y'', `m'', or `d''', InFileName, LineNo);
                Exit
              end;
            case Format[pf + 1] of
              '%': begin
                    if DateStr[pd] <> '%' then
                      Error ('date specification does not match format', InFileName, LineNo);
                    Inc (pd)
                  end;
              'Y': ExtractInteger (DateStr, pd, Year);
              'm': ExtractInteger (DateStr, pd, Month);
              'd': ExtractInteger (DateStr, pd, Day);
            end;
            Inc (pf, 2)
          end
        else
          begin
            if Format[pf] <> DateStr[pd] then
              Error ('date specification does not match format', InFileName, LineNo);
            Inc (pf);
            Inc (pd)
          end
      end;
    if not Day in [1..31] then
      Error ('month out of range', InFileName, LineNo);
    if not Month in [1..12] then
      Error ('month out of range', InFileName, LineNo)
  end;
    
  function ExtractDayNumber (const DateStr: String) = d: Integer;
  var
    Year, Month, Day, y: Integer;
  begin
    ParseDate (DateStr, DateFormat, Year, Month, Day);
    d := 0;
    for y := 1970 to Year - 1 do
      begin
        Inc (d, 365);
        if IsLeapYear (y) then
          Inc (d)
      end;
    Inc (d, GetDayOfYear (Day, Month, Year))
  end;
  
  function ParseOperations (O: String): POpRec;
  var
    Minus: Boolean;
    OpHead, OpTail, NewOp: POpRec;
    p: Integer;
    OtherAccount: PAccountRec;
  begin
    Minus := False;
    OpHead := nil;
    OpTail := nil;
    while O <> '' do
      case O[1] of
        ' ', '+': Delete (O, 1, 1);
        '-': begin
              Minus := True;
              Delete (O, 1, 1)
            end;
      otherwise
        p := 1;
        while (p <= Length (O)) and (O[p] <> ' ') do
          Inc (p);
        OtherAccount := FindAccount (Copy (O, 1, p - 1));
        if OtherAccount = nil then
          Error ('account `' + Copy (O, 1, p - 1) + ''' not found', InFileName, LineNo)
        else
          begin
            New (NewOp);
            if Minus then
              NewOp^.OpCode := '-'
            else
              NewOp^.OpCode := '+';
            NewOp^.OtherAccount := OtherAccount;
            NewOp^.Next := nil;
            if OpTail = nil then
              OpHead := NewOp
            else
              OpTail^.Next := NewOp;
            OpTail := NewOp
          end;
        Minus := False;
        Delete (O, 1, p)
      end;
    ParseOperations := OpHead
  end;
  
  procedure ReadLine (var Line: String);
  var
    NextLine: TString;
  begin
    Line := '\';
    while (Line <> '') and (Line[Length (Line)] = '\') and not EOF (InFile) do
      begin
        begin
          Inc (LineNo);
          ReadLn (InFile, NextLine)
        end;
        Delete (Line, Length (Line), 1);
        Line := Line + NextLine
      end;
    if (Line <> '') and (Line[Length (Line)] = '\') then
      begin
        Error ('end of file during line continuation', InFileName, LineNo);
        Line := ''
      end
  end;
  
  procedure ProcessLine (Line: String);
  var
    p: Integer;
    S, T: TString;
    x: Real;
    OpeningFlag, StoneAgeFlag: Boolean;
  
    procedure ExtractWordRaw (var Line, S: String);
    var
      p: Integer;
    begin
      p := Pos (' ', Line);
      if p = 0 then
        begin
          S := Line;
          Line := ''
        end
      else
        begin
          S := Copy (Line, 1, p - 1);
          Delete (Line, 1, p);
          TrimLeft (Line)
        end
    end;
  
    procedure ExtractWord (var Line, S: String);
    var
      A: PAliasRec;
    begin
      ExtractWordRaw (Line, S);
      A := Aliases;
      while A <> nil do
        with A^ do
          begin
            if (Name <> nil) and (Meaning <> nil) and (S = Name^) then
              S := Meaning^;
            A := Next
          end
    end;
  
    procedure ExtractReal (var Line: String; var x: Real);
    var
      S: TString;
      Code: Integer;
    begin
      ExtractWord (Line, S);
      Val (S, x, Code);
      if Code <> 0 then
        begin
          Error ('invalid number', InFileName, LineNo);
          x := 0
        end
    end;
  
    function StringFollows (Line: String): Boolean;
    begin
      TrimLeft (Line);
      StringFollows := (Line <> '') and (Line[1] = '"')
    end;
  
    procedure ExtractString (var Line, S: String);
    var
      p: Integer;
    begin
      TrimBoth (Line);
      if (Line = '') or (Line[1] <> '"') then
        begin
          Error ('`"'' expected', InFileName, LineNo);
          Line := '';
          S := '';
          Exit
        end;
      Delete (Line, 1, 1);
      p := Pos ('"', Line);
      while (p > 1) and (Line[p - 1] = '\') do
        begin
          Delete (Line, p - 1, 1);
          p := PosFrom ('"', Line, p)
        end;
      if p <= 0 then
        begin
          Error ('second `"'' expected', InFileName, LineNo);
          S := Line;
          Line := '';
          Exit
        end;
      S := Copy (Line, 1, p - 1);
      Delete (Line, 1, p);
      TrimLeft (Line)
    end;
  
    procedure ProcessFormatDecl (const Name, Format: String);
    begin
      if Name = 'date' then
        DateFormat := Format
      else if Name = 'TeXdecpoint' then
        DecimalPointFormat := Format
      else if Name = 'TeXthsep' then
        ThousandSepFormat := Format
      else
        Error ('unknown format specification', InFileName, LineNo)
    end;
  
    procedure ProcessAliasDecl (const Name, Meaning: String);
    var
      NewAlias: PAliasRec;
    begin
      New (NewAlias);
      NewAlias^.Name := NewString (Name);
      NewAlias^.Meaning := NewString (Meaning);
      NewAlias^.Next := Aliases;
      Aliases := NewAlias
    end;
  
    procedure ProcessDefaultVAT (x: Real);
    begin
      DefaultVAT := x
    end;
  
    function FindVATAccount (VAT: Real): PAccountRec;
    var
      A: PAccountRec;
    begin
      if VAT = 0 then
        begin
          FindVATAccount := nil;
          Exit
        end;
      A := Accounts;
      while A <> nil do
        begin
          if A^.CollectVAT = VAT then
            begin
              FindVATAccount := A;
              Exit
            end;
          A := A^.Next
        end;
      FindVATAccount := nil
    end;
  
    procedure ProcessSectionDecl (const Name: String; TeXFlag: Boolean);
    var
      NewAccount: PAccountRec;
    begin
      New (NewAccount);
      NewAccount^.LongName := NewString (Name);
      NewAccount^.ShortName := nil;
      NewAccount^.SectionFlag := not TeXFlag;
      NewAccount^.TeXFlag := TeXFlag;
      NewAccount^.OpeningFlag := False;
      NewAccount^.StoneAgeFlag := False;
      NewAccount^.CollectVAT := 0.0;
      NewAccount^.Events := nil;
      NewAccount^.Operations := nil;
      NewAccount^.Next := Accounts;
      Accounts := NewAccount
    end;
  
    procedure ProcessAccountDecl (const LongName, ShortName: String;
                                  CollectVAT: Real; OpeningFlag, StoneAgeFlag: Boolean;
                                  var RestOfLine: String);
    var
      NewAccount: PAccountRec;
      S: TString;
    begin
      if (FindAccount (ShortName) <> nil)
        or (FindAccount (LongName) <> nil) then
        Error ('account `' + LongName + ''' declared more than once', InFileName, LineNo);
      New (NewAccount);
      if LongName = '' then
        NewAccount^.LongName := nil
      else
        NewAccount^.LongName := NewString (LongName);
      if ShortName = '' then
        NewAccount^.ShortName := nil
      else
        NewAccount^.ShortName := NewString (ShortName);
      NewAccount^.SectionFlag := False;
      NewAccount^.TeXFlag := False;
      NewAccount^.OpeningFlag := OpeningFlag;
      NewAccount^.StoneAgeFlag := StoneAgeFlag;
      NewAccount^.CollectVAT := CollectVAT;
      NewAccount^.Events := nil;
      NewAccount^.Next := Accounts;
      Accounts := NewAccount;
      ExtractWord (RestOfLine, S);
      if S = '=' then
        begin
          NewAccount^.Operations := ParseOperations (RestOfLine);
          RestOfLine := ''
        end
      else
        NewAccount^.Operations := nil
    end;
  
    procedure ProcessEvent (Account, AntiAccount: PAccountRec;
                            Amount: Real; const Comment: String);
    var
      NewEvent, Event, PrevEvent: PEventRec;
    begin
      if Account = nil then
        begin
          Error ('missing account specification', InFileName, LineNo);
          Exit
        end;
      if Account^.Operations <> nil then
        begin
          Error ('direct money transfer involving a combined account does not make sense', InFileName, LineNo);
          Exit
        end;
      New (NewEvent);
      NewEvent^.Date := NewString (CurrentDate);
      NewEvent^.DayNumber := CurrentDayNumber;
      NewEvent^.Account := Account;
      NewEvent^.AntiAccount := AntiAccount;
      NewEvent^.Amount := Amount;
      NewEvent^.Comment := NewString (Comment);
      Event := Account^.Events;
      PrevEvent := nil;
      while (Event <> nil) and (Event^.DayNumber > NewEvent^.DayNumber) do
        begin
          PrevEvent := Event;
          Event := Event^.Next
        end;
      NewEvent^.Next := Event;
      if PrevEvent = nil then
        Account^.Events := NewEvent
      else
        PrevEvent^.Next := NewEvent
    end;
  
    procedure ParseEvent (VATMode: VATModeType; var Line: String);
    var
      Amount, CurrentVAT, AntiAmount, VAT: Real;
      Comment, S: TString;
      FromAccount, ForAccount, VATAccount: PAccountRec;
      FromFound, ForFound: Boolean;
    begin
      if (CurrentDayNumber < MinDayNumber)
        or (CurrentDayNumber > MaxDayNumber) then
        Exit;
      CurrentVAT := DefaultVAT;
      Comment := '';
      FromAccount := nil;
      ForAccount := nil;
      FromFound := False;
      ForFound := False;
      ExtractReal (Line, Amount);
      while Line <> '' do
        if StringFollows (Line) then
          begin
            if Comment <> '' then
              Error ('double comment', InFileName, LineNo);
            ExtractString (Line, Comment)
          end
        else
          begin
            ExtractWord (Line, S);
            if S = 'from' then
              begin
                FromFound := True;
                ExtractWord (Line, S);
                FromAccount := FindAccount (S);
                if FromAccount = nil then
                  Error ('invalid "from" account', InFileName, LineNo)
              end
            else if S = 'for' then
              begin
                ForFound := True;
                ExtractWord (Line, S);
                ForAccount := FindAccount (S);
                if ForAccount = nil then
                  Error ('invalid "for" account', InFileName, LineNo)
              end
            else if S = 'VAT' then
              ExtractReal (Line, CurrentVAT)
            else
              Error ('unrecognised keyword `' + S + '''', InFileName, LineNo)
          end;
      if not FromFound then
        Error ('missing "from" account', InFileName, LineNo);
      if not ForFound then
        Error ('missing "for" account', InFileName, LineNo);
      if (FromAccount <> nil) and (ForAccount <> nil) then
        begin
          case VATMode of
            NoVAT:     AntiAmount := Amount;
            Net:       AntiAmount := Round2 (Amount * (1 + CurrentVAT));
            Gross:     AntiAmount := Round2 (Amount / (1 + CurrentVAT));
            AntiNet:   begin
                        AntiAmount := Amount;
                        Amount := Round2 (AntiAmount * (1 + CurrentVAT))
                      end;
            AntiGross: begin
                        AntiAmount := Amount;
                        Amount := Round2 (AntiAmount / (1 + CurrentVAT))
                      end;
          end;
          ProcessEvent (FromAccount, ForAccount, -Amount, Comment);
          ProcessEvent (ForAccount, FromAccount, AntiAmount, Comment);
          VAT := Amount - AntiAmount;
          if VAT <> 0 then
            begin
              if VATMode in [Net, AntiGross] then
                VATAccount := FindVATAccount (-CurrentVAT)
              else
                VATAccount := FindVATAccount (CurrentVAT);
              if VATAccount <> nil then
                case VATMode of
                  NoVAT:     Error ('internal error: nonzero VAT in `noVAT'' event', InFileName, LineNo);
                  Net:       ProcessEvent (VATAccount, FromAccount, VAT, Comment);
                  Gross:     ProcessEvent (VATAccount, ForAccount, VAT, Comment);
                  AntiNet:   ProcessEvent (VATAccount, ForAccount, VAT, Comment);
                  AntiGross: ProcessEvent (VATAccount, FromAccount, VAT, Comment);
                end
            end
        end
    end;
  
  begin
    p := Pos ('#', Line);
    if p <> 0 then
      Delete (Line, p);
    TrimBoth (Line);
    if Line = '' then
      Exit;
    if Line[1] = '@' then
      begin
        Delete (Line, 1, 1);
        TrimLeft (Line);
        ExtractWord (Line, CurrentDate);
        CurrentDayNumber := ExtractDayNumber (CurrentDate);
        CurrentRep := '';
        CurrentRepUntil := '';
        CurrentRepCount := '';
        ExtractWord (Line, S);
        if S = 'repeat' then
          begin
            ExtractWord (Line, CurrentRep);
            ExtractWord (Line, S);
            if S = 'until' then
              ExtractWord (Line, CurrentRepUntil)
            else if S = 'for' then
              ExtractWord (Line, CurrentRepCount)
            else
              Error ('`until'' or `for'' expected', InFileName, LineNo)
          end
        else if S <> '' then
          Error ('`repeat'' expected, "' + S + '" found', InFileName, LineNo);
        Exit
      end;
    ExtractWord (Line, S);
    if S = 'include' then
      begin
        ExtractString (Line, S);
        IncludeFile (S)
      end
    else if S = 'format' then
      begin
        ExtractWord (Line, S);
        ExtractWord (Line, T);
        if T = '=' then
          ExtractString (Line, T)
        else
          Error ('`='' expected in format specification', InFileName, LineNo);
        ProcessFormatDecl (S, T);
        if Line <> '' then
          Error ('garbage after format specification', InFileName, LineNo)
      end
    else if S = 'alias' then
      begin
        ExtractWord (Line, S);
        ExtractWord (Line, T);
        if T = '=' then
          ExtractWord (Line, T)
        else
          Error ('`='' expected in alias declaration', InFileName, LineNo);
        ProcessAliasDecl (S, T);
        if Line <> '' then
          Error ('garbage after alias declaration', InFileName, LineNo)
      end
    else if S = 'default' then
      begin
        ExtractWord (Line, S);
        if S = 'VAT' then
          begin
            ExtractReal (Line, x);
            ProcessDefaultVAT (x)
          end
        else
          Error ('specifying default value for something unknown', InFileName, LineNo)
      end
    else if S = 'section' then
      begin
        ExtractString (Line, S);
        ProcessSectionDecl (S, False)
      end
    else if (S = 'TeX') or (S = 'tex') then
      begin
        ExtractString (Line, S);
        ProcessSectionDecl (S, True)
      end
    else if S = 'account' then
      begin
        x := 0;
        OpeningFlag := False;
        StoneAgeflag := False;
        ExtractWord (Line, S);
        if S = 'for' then
          begin
            ExtractWord (Line, S);
            if S = 'VAT' then
              ExtractReal (Line, x)
            else if S = 'opening' then
              OpeningFlag := True
            else
              Error ('`VAT'' or `opening'' expected after `account for''', InFileName, LineNo);
            ExtractWord (Line, S)
          end
        else if S = 'from' then
          begin
            ExtractWord (Line, S);
            if S = 'VAT' then
              ExtractReal (Line, x)
            else if S = 'stoneage' then
              StoneAgeFlag := True
            else
              Error ('`VAT'' or `stoneage'' expected after `account from''', InFileName, LineNo);
            x := -x;
            ExtractWord (Line, S)
          end;
        if Line <> '' then
          begin
            T := S;
            ExtractWord (Line, S);
            if S = '=' then
              begin
                S := T;
                T := '';
                Line := '= ' + Line
              end
          end
        else
          T := '';
        ProcessAccountDecl (S, T, x, OpeningFlag, StoneAgeFlag, Line);
        if Line <> '' then
          Error ('garbage after account declaration', InFileName, LineNo)
      end
    else if S = 'noVAT' then
      ParseEvent (NoVAT, Line)
    else if S = 'net' then
      ParseEvent (Net, Line)
    else if S = 'gross' then
      ParseEvent (Gross, Line)
    else if S = 'antinet' then
      ParseEvent (AntiNet, Line)
    else if S = 'antigross' then
      ParseEvent (AntiGross, Line)
    else
      Error ('command not recognised', InFileName, LineNo)
  end;
  
begin
  Assign (InFile, InFileName);
  Reset (InFile);
  while not EOF (InFile) do
    begin
      ReadLine (Line);
      ProcessLine (Line)
    end;
  Close (InFile);
end;

procedure ProcessOutput;
var
  OutFile: Text;
  Account: PAccountRec;

  procedure OutputData;
  var
    OutFile, GnuPlotFile: Text;
    D, O: TString;
    p, DN, MinDay, MaxDay, i: Integer;
    Account: PAccountRec;
    Event: PEventRec;
  begin
    Assign (OutFile, OutFileName);
    Rewrite (OutFile);
    D := DataSpec;
    p := Pos (',', D);
    MinDay := MaxInt;
    MaxDay := 0;
    while p > 0 do
      begin
        Account := FindAccount (Copy (D, 1, p - 1));
        if Account = nil then
          Error ('account `' + Copy (D, 1, p - 1) + ''' not found', '', 0)
        else
          begin
{
            if Account^.Operations <> nil then
              begin
                O := Account^.Operations^;
                while O <> '' do
                  begin
                    while (O <> '') and (O[1] in [' ', '+', '-']) do
                      Delete (O, 1, 1);
                    while (O <> '') and (O[1] <> ' ') do
                      begin
                        D := D + O[1];
                        Delete (O, 1, 1)
                      end;
                    D := D + ','
                  end
              end;
}
            Event := Account^.Events;
            while Event <> nil do
              begin
                if Event^.Date <> nil then
                  begin
                    MinDay := Min (MinDay, Event^.DayNumber);
                    MaxDay := Max (MaxDay, Event^.DayNumber)
                  end;
                Event := Event^.Next
              end
          end;
        Delete (D, 1, p);
        p := Pos (',', D)
      end;
    Write (OutFile, '#    Day');
    D := DataSpec;
    p := Pos (',', D);
    while p > 0 do
      begin
        Write (OutFile, Copy (D, 1, Min (10, p - 1)) : 12);
        Delete (D, 1, p);
        p := Pos (',', D)
      end;
    WriteLn (OutFile);
    for DN := MinDay to MaxDay do
      begin
        Write (OutFile, DN : 8);
        D := DataSpec;
        p := Pos (',', D);
        while p > 0 do
          begin
            Account := FindAccount (Copy (D, 1, p - 1));
            Write (OutFile, AccountSumAt (Account, DN) : 12 : 2);
            Delete (D, 1, p);
            p := Pos (',', D)
          end;
        WriteLn (OutFile)
      end;
    Close (OutFile);
    if GnuPlot then
      begin
        Assign (GnuPlotFile, GnuPlotFileName);
        ReWrite (GnuPlotFile);
        Write (GnuPlotFile, 'plot');
        D := DataSpec;
        i := 1;
        p := Pos (',', D);
        while p > 0 do
          begin
            if i > 1 then
              Write (GnuPlotFile, ',');
            Inc (i);
            WriteLn (GnuPlotFile, ' \');
            Write (GnuPlotFile, '  "', OutFileName, '" using 1:', i, ' title "',
                   Copy (D, 1, p - 1), '" with lines');
            Delete (D, 1, p);
            p := Pos (',', D)
          end;
        WriteLn (GnuPlotFile);
        WriteLn (GnuPlotFile, 'pause -1');
        Close (GnuPlotFile);
        if Execute ('gnuplot ' + GnuPlotFileName) <> 0 then
          Error ('error executing `gnuplot ' + GnuPlotFileName + '''', '', 0);
      end
  end;

  procedure OutputAccountSummary (Account: PAccountRec);
  var
    i: Integer;
    Title: TString;
  begin
    if (Account <> nil)
       and (Account^.SectionFlag
            or (Account^.Events <> nil)
            or (Account^.Operations <> nil)) then
      begin
        Title := AccountTitleStr (Account);
        Write (OutFile, Title);
        if Account^.SectionFlag then
          Write (OutFile, ':')
        else
          begin
            for i := Length (Title) + 1 to 40 do
              Write (OutFile, ' ');
            Write (OutFile, AccountSumPlusStr (Account) : 12,
                   AccountSumMinusStr (Account) : 12,
                   '  =', AccountSumStr (Account) : 12)
          end;
        WriteLn (OutFile)
      end
  end;

  procedure OutputAccountEvents (Account: PAccountRec);
  var
    i: Integer;
    Title, Line: TString;
    Op: POpRec;
    Event: PEventRec;

    function ZeroRing (AllEvents, Event: PEventRec): Boolean;
    var
      S: Real;
    begin
      if (Event = nil) or (Event^.Comment = nil) then
        begin
          ZeroRing := False;
          Exit
        end;
      S := 0.0;
      while AllEvents <> nil do
        with AllEvents^ do
          begin
            if (Comment <> nil) and (Comment^ = Event^.Comment^) then
              S := S + Amount;
            AllEvents := Next
          end;
      ZeroRing := Abs (S) < 0.005
    end;

  begin
    if (Account <> nil)
       and (Account^.SectionFlag
            or (Account^.Events <> nil)
            or (Account^.Operations <> nil)) then
      begin
        Title := AccountTitleStr (Account);
        WriteLn (OutFile, Title);
        if Account^.SectionFlag then
          for i := 1 to Length (Title) do
            Write (OutFile, '=')
        else
          for i := 1 to Length (Title) do
            Write (OutFile, '~');
        WriteLn (OutFile);
        if Account^.Operations <> nil then
          begin
            Op := Account^.Operations;
            while Op <> nil do
              begin
                if Op^.OpCode = '-' then
                  Write (OutFile, '-')
                else
                  Write (OutFile, '+');
                if Op^.OtherAccount^.ShortName <> nil then
                  Write (OutFile, Op^.OtherAccount^.ShortName^ : 7)
                else
                  Write (OutFile, '       ');
                Write (OutFile, ' ');
                if Op^.OtherAccount^.LongName <> nil then
                  Line := Copy (Op^.OtherAccount^.LongName^, 1, 23)
                else
                  Line := '';
                while Length (Line) < 27 do
                  Line := Line + ' ';
                Write (OutFile, Line);
                if Op^.OpCode = '-' then
                  Write (OutFile, AccountSumMinusStr (Op^.OtherAccount) : 12,
                                  AccountSumPlusStr (Op^.OtherAccount) : 12)
                else
                  Write (OutFile, AccountSumPlusStr (Op^.OtherAccount) : 12,
                                  AccountSumMinusStr (Op^.OtherAccount) : 12);
                WriteLn (OutFile, '    ', AccountSumStr (Op^.OtherAccount) : 12);
                Op := Op^.Next
              end
          end;
        Event := Account^.Events;
        while Event <> nil do
          with Event^ do
            begin
              if not (CurrentOnly and ZeroRing (Account^.Events, Event)) then
                begin
                  Line := '';
                  if Date <> nil then
                    WriteStr (Line, Date^ : 10);
                  while Length (Line) < 12 do
                    Line := Line + ' ';
                  if (AntiAccount <> nil)
                     and (AntiAccount^.ShortName <> nil) then
                    WriteStr (Line, Line, AntiAccount^.ShortName^ : 5);
                  while Length (Line) < 18 do
                    Line := Line + ' ';
                  if Comment <> nil then
                    Line := Line + Comment^;
                  while Length (Line) < 64 do
                    Line := Line + ' ';
                  WriteLn (OutFile, Line, Amount : 12 : 2)
                end;
              Event := Next
            end;
        if not Account^.SectionFlag then
          begin
            for i := 1 to 64 do
              Write (OutFile, ' ');
            WriteLn (OutFile, '------------');
            WriteLn (OutFile, AccountSumPlusStr (Account) : 48,
                     AccountSumMinusStr (Account) : 12,
                     '    ', AccountSumStr (Account) : 12)
          end;
        WriteLn (OutFile)
      end
  end;

  procedure OutputTeX;
  var
    OutFile: Text;
    p: Integer;
    Title: TString;
    Op: POpRec;
    Event: PEventRec;
    FirstSection: Boolean;
    Account, OtherAccount: PAccountRec;
  begin
    Assign (OutFile, OutFileName);
    Rewrite (OutFile);
    WriteLn (OutFile, '\def\bfgversion{', BFGVersion, '}');
    FirstSection := True;
    Account := Accounts;
    while Account <> nil do
      begin
        if Account^.SectionFlag
           or Account^.TeXFlag
           or (Account^.Events <> nil)
           or (Account^.Operations <> nil) then
          begin
            Title := TeXStr (AccountTitleStr (Account));
            if Account^.TeXFlag and (Account^.LongName <> nil) then
              WriteLn (OutFile, Account^.LongName^)
            else if Account^.SectionFlag then
              begin
                if not FirstSection then
                  WriteLn (OutFile, '\endbfgsection');
                WriteLn (OutFile, '\bfgsection{', Title, '}');
                FirstSection := False
              end
            else
              begin
                if Account^.StoneAgeFlag then
                  Write (OutFile, '\bfgstoneageaccount{')
                else
                  Write (OutFile, '\bfgaccount{');
                if Account^.ShortName <> nil then
                  Write (OutFile, TeXStr (Account^.ShortName^));
                Write (OutFile, '}{');
                if Account^.LongName <> nil then
                  Write (OutFile, TeXStr (Account^.LongName^));
                Write (OutFile, '}{');
                if Account^.StoneAgeFlag then
                  Write (OutFile, TeXAmount (AccountSumPlusOpeningStr (Account), True, WithMacro))
                else
                  Write (OutFile, TeXAmount (AccountSumOpeningStr (Account), False, WithMacro));
                Write (OutFile, '}{',
                       TeXAmount (AccountSumPlusStr (Account), False, WithoutSign), '}{',
                       TeXAmount (AccountSumMinusStr (Account), False, WithoutSign), '}{');
                if Account^.StoneAgeFlag then
                  Write (OutFile,
                         TeXAmount (AccountStoneAgeSumPlusStr (Account), True, WithMacro), '}{',
                         TeXAmount (AccountSumMinusOpeningStr (Account), True, WithMacro), '}{',
                         TeXAmount (AccountStoneAgeSumMinusStr (Account), True, WithMacro))
                else
                  Write (OutFile, TeXAmount (AccountSumStr (Account), True, WithMacro));
                WriteLn (OutFile, '}');
                if not SummaryOnly then
                  begin
                    if Account^.Operations <> nil then
                      begin
                        Op := Account^.Operations;
                        while Op <> nil do
                          begin
                            if Op^.OpCode = '-' then
                              Write (OutFile, '-')
                            else
                              Write (OutFile, '+');
                            Write (OutFile, '}{');
                            if Op^.OtherAccount^.ShortName <> nil then
                              Write (OutFile, TeXStr (Op^.OtherAccount^.ShortName^));
                            Write (OutFile, '}{');
                            if Op^.OtherAccount^.LongName <> nil then
                              Write (OutFile, TeXStr (Op^.OtherAccount^.LongName^));
                            Write (OutFile, '}{', TeXAmount (AccountSumOpeningStr (Op^.OtherAccount), False, WithMacro), '}{');
                            if Op^.OpCode = '-' then
                              Write (OutFile, TeXAmount (AccountSumMinusStr (Op^.OtherAccount), False, WithoutSign), '}{',
                                              TeXAmount (AccountSumPlusStr (Op^.OtherAccount), False, WithoutSign))
                            else
                              Write (OutFile, TeXAmount (AccountSumPlusStr (Op^.OtherAccount), False, WithoutSign), '}{',
                                              TeXAmount (AccountSumMinusStr (Op^.OtherAccount), False, WithoutSign));
                            WriteLn (OutFile, '}{', TeXAmount (AccountSumStr (Op^.OtherAccount), True, WithMacro), '}');
                            Op := Op^.Next
                          end
                      end;
                    Event := Account^.Events;
                    while Event <> nil do
                      with Event^ do
                        begin
                          Write (OutFile, '\bfgevent{');
                          if Date <> nil then
                            Write (OutFile, Date^);
                          Write (OutFile, '}{');
                          if (AntiAccount <> nil)
                             and (AntiAccount^.ShortName <> nil) then
                            Write (OutFile, TeXStr (AntiAccount^.ShortName^));
                          Write (OutFile, '}{');
                          if (AntiAccount <> nil)
                             and (AntiAccount^.LongName <> nil) then
                            Write (OutFile, TeXStr (AntiAccount^.LongName^));
                          Write (OutFile, '}{');
                          if Comment <> nil then
                            Write (OutFile, Comment^);
                          Write (OutFile, '}{');
                          WriteLn (OutFile, TeXAmountNum (Amount, False, WithMacro), '}');
                          Event := Next
                        end;
                    if Account^.StoneAgeFlag then
                      WriteLn (OutFile, '\endbfgstoneageaccount')
                    else
                      WriteLn (OutFile, '\endbfgaccount')
                  end
              end
          end;
        Account := Account^.Next
      end;
    if not FirstSection then
      WriteLn (OutFile, '\endbfgsection');
    Close (OutFile)
  end;

begin
  if Accounts = nil then
    Error ('no accounts', '', 0)
  else if DataSpec <> '' then
    OutputData
  else if GnuPlot then
    Error ('`--gnuplot'' requires `--data''', '', 0)
  else if TeX then
    OutputTeX
  else
    begin
      Assign (OutFile, OutFileName);
      Rewrite (OutFile);
      Account := Accounts;
      while Account <> nil do
        begin
          if SummaryOnly then
            OutputAccountSummary (Account)
          else
            OutputAccountEvents (Account);
          Account := Account^.Next
        end;
      Close (OutFile)
    end
end;

begin
  Init;
  IncludeFile (MainFileName);
  ReverseLists;
  if Force or not ErrorDetected then
    ProcessOutput;
  Done
end.
