#include <limits.h>
#include <GL/glut.h>
#include <png.h>
#include "robosim.h"

#define DELAY 100

static png_byte *picture;
static GLuint texture;
static unsigned int w, h;
static int rx, ry, dx = 0, dy = -1, moving = 0, distance = 0;
static void (*user_keyboard_handler) (char which_key) = NULL;
static void (*user_timer_handler) (void) = NULL;

static png_byte *load_png_file (char *filename, unsigned *w, unsigned *h)
{
  png_struct *png_ptr = NULL;
  png_info *info_ptr = NULL;
  png_byte buf[8];
  png_byte *png_pixels = NULL;
  png_byte **row_pointers = NULL;
  png_uint_32 row_bytes;
  png_uint_32 width;
  png_uint_32 height;
  int bit_depth;
  int color_type;
  int i;

  FILE *f = fopen (filename, "rb");

  if (!f)
    return NULL;

  /* is it a PNG file? */
  if (fread (buf, 1, 8, f) != 8)
    {
      fclose (f);
      return NULL;
    }
  if (!png_check_sig (buf, 8))
    {
      fclose (f);
      return NULL;
    }

  png_ptr = png_create_read_struct (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!png_ptr)
    {
      fclose (f);
      return NULL;
    }

  info_ptr = png_create_info_struct (png_ptr);
  if (!info_ptr)
    {
      fclose (f);
      png_destroy_read_struct (&png_ptr, NULL, NULL);
      return NULL;
    }

  if (setjmp (png_jmpbuf (png_ptr)))
    {
      fclose (f);
      png_destroy_read_struct (&png_ptr, &info_ptr, NULL);
      return NULL;
    }

  png_init_io (png_ptr, f);
  png_set_sig_bytes (png_ptr, 8);
  png_read_info (png_ptr, info_ptr);

  png_get_IHDR (png_ptr, info_ptr, &width, &height, &bit_depth,
		&color_type, NULL, NULL, NULL);

  /* tell libpng to strip 16 bit/color files down to 8 bits/color */
  if (bit_depth == 16)
    png_set_strip_16 (png_ptr);
  /* expand paletted colors into true RGB triplets */
  if (color_type == PNG_COLOR_TYPE_PALETTE)
    png_set_expand (png_ptr);
  /* expand grayscale images to the full 8 bits from 1, 2, or 4 bits/pixel */
  if (color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8)
    png_set_expand (png_ptr);
  /* expand paletted or RGB images with transparency to full alpha channels
     so the data will be available as RGBA quartets. */
  if (png_get_valid (png_ptr, info_ptr, PNG_INFO_tRNS))
    png_set_expand (png_ptr);
  /* transform grayscale images into rgb */
  if (color_type == PNG_COLOR_TYPE_GRAY ||
      color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
    png_set_gray_to_rgb (png_ptr);

  png_read_update_info (png_ptr, info_ptr);
  png_get_IHDR (png_ptr, info_ptr, &width, &height, &bit_depth,
		&color_type, NULL, NULL, NULL);
  if (color_type != PNG_COLOR_TYPE_RGB
      && color_type != PNG_COLOR_TYPE_RGB_ALPHA)
    {
      fclose (f);
      png_destroy_read_struct (&png_ptr, &info_ptr, NULL);
      return NULL;
    }

  row_bytes = png_get_rowbytes (png_ptr, info_ptr);

  png_pixels = (png_byte *) malloc (row_bytes * height * sizeof (png_byte));
  if (png_pixels == NULL)
    {
      fclose (f);
      png_destroy_read_struct (&png_ptr, &info_ptr, NULL);
      return NULL;
    }

  row_pointers = (png_byte **) malloc (height * sizeof (png_bytep));
  if (row_pointers == NULL)
    {
      fclose (f);
      png_destroy_read_struct (&png_ptr, &info_ptr, NULL);
      free (png_pixels);
      png_pixels = NULL;
      return NULL;
    }

  for (i = 0; i < height; i++)
    row_pointers[i] = png_pixels + i * row_bytes;

  png_read_image (png_ptr, row_pointers);

  png_read_end (png_ptr, info_ptr);
  png_destroy_read_struct (&png_ptr, &info_ptr, NULL);
  fclose (f);

  if (w)
    *w = width;
  if (h)
    *h = height;
  free (row_pointers);

  return png_pixels;
}

static void display_handler (void)
{
  glClear (GL_COLOR_BUFFER_BIT);
  glColor3f (1, 1, 1);
  glBegin (GL_QUADS);
  glTexCoord2f (0, 1);
  glVertex2f (0, 0);
  glTexCoord2f (1, 1);
  glVertex2f (w, 0);
  glTexCoord2f (1, 0);
  glVertex2f (w, h);
  glTexCoord2f (0, 0);
  glVertex2f (0, h);
  glEnd ();

  glBegin (GL_POLYGON);
  glColor3f (1, 0, 0);
  glVertex2f (rx - 10 * dx - 10 * dy, ry - 10 * dy + 10 * dx);
  glVertex2f (rx - 10 * dx + 10 * dy, ry - 10 * dy - 10 * dx);
  glVertex2f (rx + 10 * dx + 6 * dy, ry + 10 * dy - 6 * dx);
  glVertex2f (rx + 10 * dx - 6 * dy, ry + 10 * dy + 6 * dx);
  glEnd ();

  glFlush ();
  glutSwapBuffers ();
}

static int is_wall (void)
{
  png_byte *p = picture + 4 * (w * (h - ry) + rx);
  return *p++ < 0x80 && *p++ < 0x80 && *p < 0x80;
}

static int is_target (void)
{
  png_byte *p = picture + 4 * (w * (h - ry) + rx);
  return *p++ < 0x80 && *p++ > 0x80 && *p < 0x80;
}

static void keyboard_handler (unsigned char key, int x, int y)
{
  if (user_keyboard_handler)
    user_keyboard_handler (key);
}

static void timer_handler (int value)
{
  if (moving)
    {
      rx += dx;
      ry += dy;
      distance++;
      glutPostRedisplay ();
    }
  if (user_timer_handler)
    {
      user_timer_handler ();
      glutPostRedisplay ();
    }
  glutTimerFunc (DELAY, timer_handler, 0);
}

void robosim_init (int argc, char **argv, char *png_filename,
                   int initial_x, int initial_y, int initial_direction)
{
  if (png_filename == NULL)
    {
      fprintf (stderr, "robosim: need an image filename\n");
      exit (1);
    }
  picture = load_png_file (png_filename, &w, &h);
  if (picture == NULL)
    {
      fprintf (stderr, "robosim: cannot load image \"%s\"\n", png_filename);
      exit (1);
    }

  glutInit (&argc, argv);
  glutInitWindowSize (w, h);
  glutInitDisplayMode (GLUT_RGB | GLUT_DOUBLE);
  (void) glutCreateWindow ("RoboSim");

  glViewport (0, 0, w, h);
  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  glOrtho (0, w, 0, h, -1, 1);
  glMatrixMode (GL_MODELVIEW);
  glLoadIdentity ();
  glClearColor (0, 0, 0, 0);
  glEnable (GL_TEXTURE_2D);

  glGenTextures (1, &texture);
  glBindTexture (GL_TEXTURE_2D, texture);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  gluBuild2DMipmaps (GL_TEXTURE_2D, 3, w, h, GL_RGBA, GL_UNSIGNED_BYTE, picture);

  rx = initial_x;
  ry = initial_y;
  robosim_rotate (initial_direction);

  glutDisplayFunc (display_handler);
  glutKeyboardFunc (keyboard_handler);
  glutTimerFunc (DELAY, timer_handler, 0);

  glutPostRedisplay ();
}

void robosim_run (void)
{
  glutMainLoop ();
}

void robosim_start_move (void)
{
  moving = 1;
}

void robosim_stop_move (void)
{
  moving = 0;
}

int robosim_moving (void)
{
  return moving;
}

int robosim_get_distance (void)
{
  return distance;
}

void robosim_reset_distance (void)
{
  distance = 0;
}

void robosim_rotate (int degrees)
{
  int tmp;
  switch (degrees)
  {
  case 0:
  case 360:
  case -360:
    /* do nothing */
    break;
  case 180:
  case -180:
    dx = -dx;
    dy = -dy;
    break;
  case 90:
  case -270:
    tmp = dy;
    dy = -dx;
    dx = tmp;
    break;
  case -90:
  case 270:
    tmp = dy;
    dy = dx;
    dx = -tmp;
    break;
  default:
    fprintf (stderr, "robosim: degrees out of square\n");
    exit (1);
  }
  glutPostRedisplay ();
}

robosim_sensor_result robosim_query_sensor (void)
{
  if (is_wall ())
    return found_wall;
  else if (is_target ())
    return found_target;
  else
    return found_nothing;
}

void robosim_on_key_pressed (void (*handler) (char which_key))
{
  user_keyboard_handler = handler;
}

void robosim_on_timer (void (*handler) (void))
{
  user_timer_handler = handler;
}
