#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include "robosim.h"

#define PIXEL_SIZE 40
#define AUTO_RESTART_DELAY 10
#define DIRECTION_FORWARD   1
#define DIRECTION_BACKWARD -1

#define MARK_BUFFER_WIDTH 40
#define MARK_BUFFER_HEIGHT 30
#define MAX_PATH_LENGTH (MARK_BUFFER_WIDTH + MARK_BUFFER_HEIGHT)

#define MARK_UNKNOWN   ' '
#define MARK_REACHABLE '.'
#define MARK_KNOWN     'o'
#define MARK_WALL      '#'

#define PATH_UNKNOWN INT_MAX

int direction, distance_from_home;
int rx = MARK_BUFFER_WIDTH / 2, ry = MARK_BUFFER_HEIGHT / 2;
int tx = -1, ty = -1, dx = 1, dy = 0;
int auto_mode = 0, auto_restart_countdown = 0;
char mark_buffer[MARK_BUFFER_HEIGHT][MARK_BUFFER_WIDTH];

typedef struct
{
  int x, y;
}
path_element;

int path_buffer[MARK_BUFFER_HEIGHT][MARK_BUFFER_WIDTH];
path_element path_stack[MAX_PATH_LENGTH];
int path_stack_pointer = 0;
path_element best_path[MAX_PATH_LENGTH];
int best_path_cost = PATH_UNKNOWN, best_path_length = 0, best_path_pointer = 0;

void set_mark_buffer (int x, int y, char value)
{
  if (x < 0 || x >= MARK_BUFFER_WIDTH || y < 0 || y >= MARK_BUFFER_HEIGHT)
    {
      fprintf (stderr, "buffer overrun\n");
      exit (1);
    }
  mark_buffer[y][x] = value;
}

char get_mark_buffer (int x, int y)
{
  if (x < 0 || x >= MARK_BUFFER_WIDTH || y < 0 || y >= MARK_BUFFER_HEIGHT)
    {
      fprintf (stderr, "buffer overrun\n");
      exit (1);
    }
  return mark_buffer[y][x];
}

void mark_as_wall (int x, int y)
{
  set_mark_buffer (x, y, MARK_WALL);
}

void mark_as_known (int x, int y)
{
  set_mark_buffer (x, y, MARK_KNOWN);
}

void mark_as_reachable (int x, int y)
{
  if (get_mark_buffer (x, y) == MARK_UNKNOWN)
    set_mark_buffer (x, y, MARK_REACHABLE);
}

void mark_neighbours_as_reachable (int x, int y)
{
  mark_as_reachable (x - 1, y);
  mark_as_reachable (x + 1, y);
  mark_as_reachable (x, y - 1);
  mark_as_reachable (x, y + 1);
}

void erase_mark_buffer (void)
{
  int x, y;
  for (y = 0; y < MARK_BUFFER_HEIGHT; y++)
    for (x = 0; x < MARK_BUFFER_WIDTH; x++)
      set_mark_buffer (x, y, MARK_UNKNOWN);
}

void set_path_buffer (int x, int y, int value)
{
  if (x < 0 || x >= MARK_BUFFER_WIDTH || y < 0 || y >= MARK_BUFFER_HEIGHT)
    {
      fprintf (stderr, "buffer overrun\n");
      exit (1);
    }
  path_buffer[y][x] = value;
}

int get_path_buffer (int x, int y)
{
  if (x < 0 || x >= MARK_BUFFER_WIDTH || y < 0 || y >= MARK_BUFFER_HEIGHT)
    {
      fprintf (stderr, "buffer overrun\n");
      exit (1);
    }
  return path_buffer[y][x];
}

void erase_path_buffer (void)
{
  int x, y;
  for (y = 0; y < MARK_BUFFER_HEIGHT; y++)
    for (x = 0; x < MARK_BUFFER_WIDTH; x++)
      set_path_buffer (x, y, PATH_UNKNOWN);
}

void push_path (int x, int y)
{
  path_stack[path_stack_pointer].x = x;
  path_stack[path_stack_pointer].y = y;
  path_stack_pointer++;
}

void pop_path (void)
{
  path_stack_pointer--;
}

void rotate (int degrees)
{
  int tmp;
  switch (degrees)
  {
  case 0:
  case 360:
  case -360:
    /* do nothing */
    break;
  case 180:
  case -180:
    dx = -dx;
    dy = -dy;
    break;
  case 90:
  case -270:
    tmp = dy;
    dy = -dx;
    dx = tmp;
    break;
  case -90:
  case 270:
    tmp = dy;
    dy = dx;
    dx = -tmp;
    break;
  }
  robosim_rotate (degrees);
}

void dump_mark_buffer (void)
{
  int x, y;
  for (x = 0; x < MARK_BUFFER_WIDTH; x++)
    printf ("--");
  printf ("\n");
  for (y = MARK_BUFFER_HEIGHT - 1; y >= 0; y--)
    {
      for (x = 0; x < MARK_BUFFER_WIDTH; x++)
        if (x == rx && y == ry)
          printf (" x");
        else if (x == tx && y == ty)
          printf (" T");
        else
          printf (" %c", get_mark_buffer (x, y));
      printf ("\n");
    }
}

void dump_path_buffer (void)
{
  int x, y;
  for (x = 0; x < MARK_BUFFER_WIDTH; x++)
    printf ("--");
  printf ("\n");
  for (y = MARK_BUFFER_HEIGHT - 1; y >= 0; y--)
    {
      for (x = 0; x < MARK_BUFFER_WIDTH; x++)
        if (x == rx && y == ry)
          printf (" x");
        else if (get_path_buffer (x, y) == PATH_UNKNOWN)
          printf ("  ");
        else
          printf ("%2d", get_path_buffer (x, y));
      printf ("\n");
    }
}

void show_best_path (void)
{
  int x, y;
  for (x = 0; x < MARK_BUFFER_WIDTH; x++)
    printf ("--");
  printf ("\n");
  for (y = MARK_BUFFER_HEIGHT - 1; y >= 0; y--)
    {
      for (x = 0; x < MARK_BUFFER_WIDTH; x++)
        if (x == rx && y == ry)
          printf (" x");
        else if (get_path_buffer (x, y) == PATH_UNKNOWN)
          printf ("  ");
        else
          {
            int k, in_best_path = 0;
            for (k = 0; k < best_path_length; k++)
              if (x == best_path[k].x && y == best_path[k].y)
                {
                  in_best_path = 1;
                  break;
                }
            if (in_best_path)
              printf (" *");
            else if (get_mark_buffer (x, y) != MARK_REACHABLE)
              printf ("  ");
            else
              printf ("%2d", get_path_buffer (x, y));
          }
      printf ("\n");
    }
}

void start_move (void)
{
  direction = DIRECTION_FORWARD;
  distance_from_home = 0;
  robosim_reset_distance ();
  robosim_start_move ();
}

void key_handler (char which_key)
{
  switch (which_key)
    {
    case 'a':
      auto_mode = !auto_mode;
      auto_restart_countdown = 0;
      break;
    case 'l':
      if (!robosim_moving ())
        rotate (-90);
      break;
    case 'r':
      if (!robosim_moving ())
        rotate (90);
      break;
    case ' ':
      if (!robosim_moving ())
        start_move ();
      break;
    case 'b':
      dump_mark_buffer ();
      break;
    case 'p':
      dump_path_buffer ();
      break;
    case 'P':
      show_best_path ();
      break;
    case 'q':
      exit (0);
    }
}

void examine_path (int x, int y, int dx, int dy, int cost)
{
  if (get_mark_buffer (x, y) != MARK_WALL
      && get_mark_buffer (x, y) != MARK_UNKNOWN
      && cost < get_path_buffer (x, y))
    {
      if (cost)
        push_path (x, y);
      set_path_buffer (x, y, cost);
      if (get_mark_buffer (x, y) == MARK_REACHABLE && cost < best_path_cost)
        {
          int k;
          for (k = 0; k < path_stack_pointer; k++)
            best_path[k] = path_stack[k];
          best_path_length = path_stack_pointer;
          best_path_cost = cost;
        }
      examine_path (x + dx, y + dy, dx, dy, cost + 1);
      examine_path (x - dx, y - dy, -dx, -dy, cost + 2);
      examine_path (x + dy, y - dx, dy, -dx, cost + 2);
      examine_path (x - dy, y + dx, -dy, dx, cost + 2);
      if (cost)
        pop_path ();
    }
}

void aim_at_closest_reachable_pixel ()
{
  if (best_path_length <= 0)
    {
      best_path_cost = PATH_UNKNOWN;
      erase_path_buffer ();
      examine_path (rx, ry, dx, dy, 0);
    }
  if (best_path_length > 0)
    {
      int bdx = best_path[best_path_pointer].x - rx;
      int bdy = best_path[best_path_pointer].y - ry;
      if (bdx == -dx && bdy == -dy)
        rotate (180);
      else if (bdx == dy && bdy == -dx)
        rotate (90);
      else if (bdx == -dy && bdy == dx)
        rotate (-90);
      best_path_pointer++;
      if (best_path_pointer >= best_path_length)
        best_path_pointer = best_path_length = 0;
    }
  else
    auto_mode = 0;
}

void timer_handler (void)
{
  robosim_sensor_result sensor = robosim_query_sensor ();
  if (sensor == found_wall)
    {
      mark_as_wall (rx + dx, ry + dy);
      distance_from_home = robosim_get_distance ();
      robosim_reset_distance ();
      direction = DIRECTION_BACKWARD;
      rotate (180);
    }
  if (sensor == found_target)
    {
      tx = rx;
      ty = ry;
    }
  if (robosim_moving ())
    {
      if (direction == DIRECTION_FORWARD
          && robosim_get_distance () >= PIXEL_SIZE)
        {
          robosim_stop_move ();
          rx += dx;
          ry += dy;
          mark_as_known (rx, ry);
          mark_neighbours_as_reachable (rx, ry);
          aim_at_closest_reachable_pixel ();
        }
      if (direction == DIRECTION_BACKWARD
          && robosim_get_distance () >= distance_from_home)
        {
          robosim_stop_move ();
          aim_at_closest_reachable_pixel ();
        }
    }
  else if (auto_mode)
    {
      auto_restart_countdown--;
      if (auto_restart_countdown <= 0)
        {
          auto_restart_countdown = AUTO_RESTART_DELAY;
          start_move ();
        }
    }
}

int main (int argc, char **argv)
{
  robosim_init (argc, argv, "parcour.png", 426, 493, -90);
  robosim_on_key_pressed (key_handler);
  robosim_on_timer (timer_handler);
  erase_mark_buffer ();
  erase_path_buffer ();
  mark_as_known (rx, ry);
  mark_neighbours_as_reachable (rx, ry);
  aim_at_closest_reachable_pixel ();
  robosim_run ();
  return 0;
}
